<?php

namespace App\Http\Controllers\Admin\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Arr;
use App\Models\User;
use App\Models\Company;
use App\Models\UserBalanceHistory;

class UsersController extends Controller
{
    public function index(Request $request)
    {
        // 添加允许排序的字段白名单
        $allowedSortFields = ['id', 'email', 'created_at', 'status', 'balance'];
        $sortBy = in_array($request->input('sortBy'), $allowedSortFields)
            ? $request->input('sortBy')
            : 'created_at';

        // 获取查询参数
        $searchQuery = $request->input('q', '');
        $page = $request->input('page', 1);
        $itemsPerPage = $request->input('itemsPerPage', 10);
        $orderBy = $request->input('orderBy', 'desc');

        // 查询用户数据，并加载关联的公司数据
        $query = User::with('company'); // Eager Load 'company' 关系

        // 如果有搜索查询，则按条件过滤（根据实际需要修改字段）
        if ($searchQuery) {
            $query->where(function ($q) use ($searchQuery) {
                $q->where('email', 'like', '%' . $searchQuery . '%')
                    ->orWhere('id', 'like', '%' . $searchQuery . '%')
                    ->orWhere('status', 'like', '%' . $searchQuery . '%')
                    ->orWhere('created_at', 'like', '%' . $searchQuery . '%')
                    ->orWhere('balance', $searchQuery);
            });
        }

        // $query->whereNot('status', 'Disabled');
        $query->where('is_deleted', '0');
        // 进行排序
        $query->orderBy($sortBy, $orderBy);

        // 获取分页数据
        $users = $query->paginate($itemsPerPage, ['*'], 'page', $page);

        // 格式化用户数据，包含公司名称
        $formattedUsers = $users->map(function ($user) {
            return [
                'id' => $user->id,
                'email' => $user->email,
                'group_name' => $user->group_name,
                'status' => $user->status,
                'created_at' => $user->created_at,
                'balance' => number_format($user->balance, 2),
                'company_name' => $user->company ? $user->company->name : null, // 获取公司名称
            ];
        });

        // 返回响应数据
        return response()->json([
            'users' => $formattedUsers,
            'total' => $users->total(),
            'current_page' => $users->currentPage(),
            'per_page' => $users->perPage()
        ]);
    }

    public function getUsers()
    {
        $users = User::with('company')->where('status', 'Active')->get();

        // 使用集合方法优化
        $users = $users->map(function ($user) {
            // $user['name'] = "{$user->name} ({$user->company->name})";
            // return $user;
            if ($user->company) {
                $user['name'] = "{$user->name} ({$user->company->name})";
            } else {
                $user['name'] = "{$user->name} (未分配公司)";
            }
            return $user;
        });

        return response()->json([
            'users' => $users,
        ]);
    }

    public function getUser($id)
    {
        $user = User::with('company')->where('id', $id)->first();
        $companies = Company::Select('id', 'name')->get();
        $user->password = '';
        $user->balance = number_format($user->balance, 2);

        return response()->json([
            'user' => $user,
            'companies' => $companies
        ]);
    }

    public function addUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                Rule::unique('users'),
            ],
            'status' => 'required|in:Pending Approval,Active,Disabled',
            'price_increase' => 'required|numeric',
            'password' => [
                'required',
                'min:8',
                'regex:/[a-z]/',      // at least one lowercase letter
                'regex:/[A-Z]/',      // at least one uppercase letter
                'regex:/[0-9]/',      // at least one number
                'regex:/[@$!%*#?&]/', // at least one special character
            ],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Create user
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'price_increase' => $request->price_increase,
                'group_id' => 2,
                'group_name' => 'Company Customer',
                'company_id' => $request->user()->company_id,
                'password' => Hash::make($request->password),
                'status' => $request->status,
            ]);

            return response()->json([
                'message' => 'User created successfully',
                'data' => $user
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to create user',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function updateUser(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                Rule::unique('users')->ignore($id),
            ],
            'group_id' => 'required|integer',
            'company_id' => 'required|integer|exists:companies,id',
            'status' => 'required|in:Pending Approval,Active,Disabled',
            'password' => [
                'nullable',
                'min:8',
                'regex:/[a-z]/',      // at least one lowercase letter
                'regex:/[A-Z]/',      // at least one uppercase letter
                'regex:/[0-9]/',      // at least one number
                'regex:/[@$!%*#?&]/', // at least one special character
            ],
            'permissions' => 'array',
        ]);

        // 修复这里：先获取权限数组，然后修改它
        $permissions = $request->input('permissions', []);
        if ($request->group_id < 10 && !in_array('hide_price', $permissions)) {
            $permissions[] = 'hide_price';
        }
        // 将修改后的权限数组合并回请求
        $request->merge(['permissions' => $permissions]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::findOrFail($id);

        $groups = ['0' => 'Pending', '1' => 'Company User', '10' => 'Company Admin', '99' => 'Super Admin'];

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'group_id' => $request->group_id,
            'group_name' => $groups[(string)$request->group_id],
            'company_id' => $request->company_id,
            'permissions' => Arr::wrap($request->permissions),
            'status' => $request->status,
        ]);

        if (!empty($request->password)) {
            // 使用 Hash 加密新密码
            $user->password = Hash::make($request->password);
            $user->save();
        }

        return response()->json([
            'status' => 'success',
            'message' => 'User updated successfully',
            'user' => $user
        ]);
    }

    public function updateUserBalance(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::findOrFail($id);

        if ($user->group_id < 10) {
            throw new \Exception('User is not a company admin');
        }

        $beforeBalance = $user->balance;

        $user->balance += $request->amount;

        $user->save();

        UserBalanceHistory::create([
            'user_id' => $user->id,
            'admin_id' => Auth::id(),
            'before_balance' => $beforeBalance,
            'increase_amount' => $request->amount,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Balance updated successfully',
            'balance' => number_format($user->balance, 2)
        ]);
    }
    public function deleteUser($id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User deleted fail',
            ]);
        }
        $ret = $user->where('id', $id)->update([
            'is_deleted' => 1,
            'email' => $user->email . '@'
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully',
        ]);
    }
}
