<?php

namespace App\Http\Controllers\Checkout;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Cart;
use App\Models\Product;
use App\Models\User;
use App\Models\Discount;
use DB;
use App\Services\JsonService;

class CartController extends Controller
{
    /**
     * 添加商品到购物车
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addToCart(Request $request)
    {
        // 验证请求的数据
        $validatedData = $request->validate([
            'main_product_name' => 'required|string|max:255',
            'product_id' => 'required|integer',
            'product_name' => 'required|string|max:255',
            'product_image' => 'nullable|string|max:255',
            'quantity' => 'required|integer|min:1',
            'width' => 'nullable|numeric',
            'height' => 'nullable|numeric',
            'price' => 'required|numeric',
            'type' => 'nullable|string|max:32',
            'option' => 'nullable|json',
        ]);

        $product = Product::find($validatedData['product_id']);

        /*if($validatedData['price'] < 5) {
            return response()->json(['error' => "The minimum amount for the product is $5.00!"], 500);
        }*/

        if($product->price > $validatedData['price']) {
            return response()->json(['error' => "The product price has changed. Please refresh the page and place your order again!"], 500);
        }

        if($product->quantity < $validatedData['quantity']) {
            return response()->json(['error' => "Insufficient stock for product {$validatedData['main_product_name']} {$validatedData['product_name']}: requested {$validatedData['quantity']}, available {$product->quantity}"], 500);
        }

        // 获取当前认证用户的 ID
        $userId = Auth::id();

        $jsonService = new JsonService();

        $validatedData['option'] = $jsonService->normalizeJson($validatedData['option']);

        // 查找现有购物车项
        $existingCartItem = Cart::findSameProduct($userId, $validatedData);
        
        DB::beginTransaction();
        try {
            if (!empty($existingCartItem->id)) {
                // 更新数量
                $existingCartItem->quantity += $validatedData['quantity'];

                $discountPrice = Discount::where('product_id', $product->id)
                    ->where('min_quantity', '<=', $existingCartItem->quantity)
                    ->orderBy('min_quantity', 'desc') 
                    ->value('discount_price');  

                if($discountPrice) {
                    $existingCartItem->price = $discountPrice;
                }

                $existingCartItem->save();
                
            } else {
                $discountPrice = Discount::where('product_id', $product->id)
                    ->where('min_quantity', '<=', $validatedData['quantity'])
                    ->orderBy('min_quantity', 'desc') 
                    ->value('discount_price');  

                if($discountPrice) {
                    $validatedData['price'] = $discountPrice;
                }

                Cart::create([
                    'user_id' => $userId,
                    'main_product_name' => $validatedData['main_product_name'],
                    'product_id' => $validatedData['product_id'],
                    'product_name' => $validatedData['product_name'],
                    'product_image' => $validatedData['product_image'],
                    'quantity' => $validatedData['quantity'],
                    'width' => $validatedData['width'],
                    'height' => $validatedData['height'],
                    'price' => $validatedData['price'],
                    'type' => $validatedData['type'],
                    'option' => $validatedData['option'],
                ]);
            }
            
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Product added to cart successfully!',
            ], 201);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'Failed to add product to cart'], 500);
        }
    }

    public function getCartList()
    {
        // 获取当前已认证用户的 ID
        $userId = Auth::id();

        // 检查用户是否已登录
        if (!$userId) {
            return response()->json([
                'success' => false,
                'message' => 'User not authenticated'
            ], 401);
        }

        $data = array();

        // 查询当前用户的购物车列表
        $cartItems = Cart::where('user_id', $userId)->get();

        foreach($cartItems as $key => $cartItem) {
            $data[$key] = $cartItem;
            if($cartItem['type'] == 'custom_size') {
                $data[$key]['price'] = max(config('product.min_price'), round($cartItem['price'] * $cartItem['width'] * $cartItem['height'] / 144, 2));
                $data[$key]['total'] = $data[$key]['price'] * $cartItem['quantity'];
            } else {
                $data[$key]['total'] = $cartItem['price'] * $cartItem['quantity'];
            }
        }

        // 返回 JSON 响应
        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    public function removeFromCart($cart_id)
    {
        // 获取当前已认证用户的 ID
        $userId = Auth::id();

        // 查询指定的购物车项
        $cartItem = Cart::where('id', $cart_id)->where('user_id', $userId)->first();

        // 检查购物车项是否存在
        if (!$cartItem) {
            return response()->json([
                'success' => false,
                'message' => 'Cart item not found or does not belong to the user'
            ], 404);
        }

        // 删除购物车项
        $cartItem->delete();

        return response()->json([
            'success' => true,
            'message' => 'Cart item removed successfully'
        ]);
    }

    public function updateCartItem(Request $request, $cartId)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:1'
        ]);

        $cartItem = Cart::where('id', $cartId)->where('user_id', auth()->id())->first();

        if (!$cartItem) {
            return response()->json(['success' => false, 'message' => 'Cart item not found'], 404);
        }

        $cartItem->quantity = $validated['quantity'];
        $cartItem->save();

        return response()->json(['success' => true, 'message' => 'Cart item updated successfully']);
    }

    public function getUserBalance()
    {
        // 获取当前已认证用户的 ID
        $user = Auth::user();
        $companyAdmin = User::where('company_id', $user->company_id)->where('group_id', 10)->first();

        // 检查用户是否已登录
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not authenticated'
            ], 401);
        }

        // 返回用户信息
        return response()->json([
            'success' => true,
            'balance' => $companyAdmin->balance ?? 0,
        ]);
    }
}