<?php

namespace App\Http\Controllers\Checkout;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Cart;
use App\Models\Order;
use App\Models\Product;
use App\Models\InventoryProduct;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use DB;
use Log;
use App\Services\AuthorizeNetPaymentService;
use App\Services\EmailService;

class ConfirmController extends Controller
{
    protected $paymentService;

    public function __construct(AuthorizeNetPaymentService $paymentService)
    {
        $this->paymentService = $paymentService;
    }

    public function addOrder(Request $request)
    {
        // Validate the incoming request data
        $validatedData = $request->validate([
            'total' => 'required|numeric',
            'balance' => 'required|numeric',
            'payment_method' => 'required|string',
            'payment_information' => 'nullable',
            'status' => 'required|string'
        ]);

        try{
            $orderItems = [];
            $order = [];

            DB::transaction(function () use (&$orderItems, &$order, $validatedData) {
                $user = Auth::user();
                $companyAdmin = User::where('company_id', $user->company_id)->where('group_id', 10)->first();
                
                if($companyAdmin->balance < $validatedData['total']) {
                    throw new \Exception('Insufficient balance to complete this transaction.');
                }

                // 锁定当前用户的所有购物车条目
                $carts = Cart::where('user_id', Auth::id())->lockForUpdate()->get();
                
                if ($carts->isEmpty()) {
                    throw new \Exception('Cart is empty.');
                }

                $cartTotal = 0;

                // 首先验证所有商品
                foreach ($carts as $cart) {
                    $product = Product::lockForUpdate()->find($cart->product_id);
                    
                    if (!$product) {
                        throw new \Exception("Product {$cart->main_product_name} {$cart->product_name} not found.");
                    }
                    
                    if ($product->quantity < $cart->quantity) {
                        throw new \Exception("Insufficient stock for product {$cart->main_product_name} {$product->name}: requested {$cart->quantity}, available {$product->quantity}");
                    }

                    if($cart->type == 'custom_size') {
                        $cartTotal += max(config('product.min_price'), round($cart->price * $cart->width * $cart->height / 144, 2)) * $cart->quantity;
                    } else {
                        $cartTotal += $cart->price * $cart->quantity;
                    }
                    
                    $orderItems[] = [
                        'product' => $product,
                        'cart' => $cart
                    ];
                }

                if (bccomp((string)$cartTotal, (string)$validatedData['total'], 2) !== 0) {
                    return response()->json([
                        'error' => true,
                        'message' => 'Total amount invalid!'
                    ], 400);
                }

                $paymentSuccess = false;

                if($validatedData['payment_method'] == 'balance') {
                     $paymentSuccess = true;
                }

                if($validatedData['payment_method'] == 'card') {
                    $result = $this->paymentService->processPayment(
                        $validatedData['payment_information']['cardNumber'],
                        $validatedData['payment_information']['cardExpiry'],
                        $validatedData['payment_information']['cardCvv'],
                        round($validatedData['total'] - $validatedData['balance'], 2)
                    );

                    if (isset($result['success']) && $result['success']) {
                        $paymentSuccess = true;
                    } else {
                        throw new \Exception($result['error']);
                    }
                }

                /*if($validatedData['payment_method'] == 'bank-transfer') {
                    $paymentSuccess = true;
                }*/

                if($paymentSuccess) {
                    $order = Order::create([
                        'user_id' => $user->id, // Get the authenticated user's ID
                        'user_email' => $user->email,
                        'company_id' => $user->company_id, 
                        'total' => $validatedData['total'],
                        'use_balance' => $validatedData['balance'],
                        'payment_method' => $validatedData['payment_method'],
                        'payment_information' => json_encode($validatedData['payment_information']) ?? null,
                        'status' => 'Pending',
                    ]);

                    $products = '';

                    // 所有验证通过后，再进行库存扣减
                    foreach ($orderItems as $item) {
                        $product = $item['product'];
                        $cart = $item['cart'];
                        
                        $product->quantity -= $cart->quantity;
                        $product->save();

                        InventoryProduct::create([
                            'order_id' => $order->id,
                            'user_id' => $user->id, 
                            'company_id' => $user->company_id, 
                            'main_product_id' => $product->main_product_id,
                            'main_product_name' => $product->main_product_name,
                            'product_id' => $product->id,
                            'product_name' => $product->name,
                            'product_image' => $cart->product_image,
                            'width' => $cart->width,
                            'height' => $cart->height,
                            'price' => $cart->price,
                            'type' => $cart->type,
                            'option' => $cart->option,
                            'inventory_type' => 'in',
                            'quantity' => $cart->quantity,
                        ]);


                        $products .= $product->main_product_name . ' ' . $this->formatSize($cart->width) . '" x ' . $this->formatSize($cart->height)  . '"  qty:' . $cart->quantity . "\n";
                    }

                    if($validatedData['balance']) {
                        $companyAdmin->balance -= $validatedData['balance'];
                        $companyAdmin->save();
                    }  
                    
                    // 清空购物车
                    Cart::where('user_id', $user->id)->delete();

                    $order->status = 'Completed';
                    $order->save();

                    $order['products'] = $products;

                    $emailService = new EmailService();
                    $response = $emailService->sendNotification($order);
                }
            });
            
            return response()->json([
                'success' => true,
                'message' => 'Order created successfully',
                'order' => $order,
            ], 201);
        } catch (\Exception $e) {
            \Log::error('Checkout failed: ' . $e->getMessage(), [
                'user_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'error' => $e->getMessage(),
                'message' => $e->getMessage() ?? 'Checkout failed. Please try again.'
            ], 400);
        }
            
        
    }

    public function upload(Request $request)
    {
        $validatedData = $request->validate([
            'receipt' => 'required|image|mimes:jpeg,png,jpg,gif|max:20480',
            'order_id' => 'required||exists:orders,id'
        ]);

        try {
            // 生成唯一文件名
            $fileName = Str::uuid() . '.' . $request->file('receipt')->getClientOriginalExtension();
            
            // 存储图片
            $path = $request->file('receipt')->storeAs('receipts', $fileName, 'public');

            $order = Order::find($request->order_id);
            $order->bank_transfer_receipt = $path;
            $order->save();

            return response()->json([
                'success' => true,
                'image_url' => Storage::url($path),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    public function formatSize($number) {
        if (fmod($number, 1) === 0.00) {
            return (int) $number;
        } else {
            return $number;
        }
    }

}