<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Models\User;
use App\Models\FactoryCart;
use App\Models\InventoryProduct;
use App\Models\CartImage;
use App\Models\FactoryOrderBarcode;
use App\Models\Product;
use App\Models\MainProduct;
use DB;
use App\Services\JsonService;
use App\Services\LogisticsService;
use App\Services\V2\ShipstationService;
use App\Services\PdfImageProcessorService;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;
use App\Services\MingtengDeliveryService;

class FactoryCartController extends Controller
{
    public function sendToPrint(Request $request)
    {
        // 验证请求的数据
        $validatedData = $request->validate([
            'main_product_id' => 'required|integer',
            'main_product_name' => 'required|string|max:255',
            'product_id' => 'required|integer',
            'product_name' => 'required|string|max:255',
            'product_image' => 'nullable|string|max:255',
            'quantity' => 'required|integer|min:1',
            'width' => 'nullable|numeric',
            'height' => 'nullable|numeric',
            'price' => 'required|numeric',
            'type' => 'nullable|string|max:32',
            'option' => 'nullable|json',
        ]);

        $user = Auth::user();
        $userId = $user->id;

        $jsonService = new JsonService();

        $inventoryProductTotal = inventoryProduct::findInventoryProductsTotal($validatedData);

        $cartProductTotal = FactoryCart::where('user_id', $userId)
            ->where('product_id', $validatedData['product_id'])
            ->where('width', $validatedData['width'])
            ->where('height', $validatedData['height'])
            ->sum('quantity');

        $cartProductTotal += $validatedData['quantity'];

        $jsonService = new JsonService();

        $validatedData['option'] = $jsonService->normalizeJson($validatedData['option']);

        try {
            $product = Product::findOrFail($validatedData['product_id']);

            if (($product->status != 'In Stock' && $cartProductTotal > $inventoryProductTotal) || $product->quantity < ($cartProductTotal - $inventoryProductTotal)) {
                throw new \Exception($product->main_product_name . ' ' . $product->name . ' is out of stock.');
            }

            FactoryCart::create([
                'user_id' => $userId,
                'main_product_id' => $validatedData['main_product_id'],
                'main_product_name' => $validatedData['main_product_name'],
                'product_id' => $validatedData['product_id'],
                'product_name' => $validatedData['product_name'],
                'product_image' => $validatedData['product_image'],
                'quantity' => $validatedData['quantity'],
                'width' => $validatedData['width'],
                'height' => $validatedData['height'],
                'price' => $product['price'],
                'type' => $validatedData['type'],
                'option' => $validatedData['option'],
                'sort_order' => (FactoryCart::where('user_id', $userId)->max('sort_order') ?? 0) + 1
            ]);

            $insufficientStock = false;

            if ($inventoryProductTotal <  $cartProductTotal) {
                $insufficientStock = "Insufficient stock for product {$validatedData['main_product_name']} {$validatedData['product_name']}: requested {$cartProductTotal}, available {$inventoryProductTotal}, insufficient stock will be purchased using the balance.\n";
            }

            return response()->json([
                'success' => true,
                'message' => 'Product sent to print successfully!',
                'insufficient_stock' => $insufficientStock
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage() ?? 'Failed to send product to print.',
                'message' => $e->getMessage() ?? 'Failed to send product to print.'
            ], 500);
        }
    }

    public function getFactoryCartList()
    {
        // 获取当前已认证用户的 ID
        $userId = Auth::id();

        // 检查用户是否已登录
        if (!$userId) {
            return response()->json([
                'success' => false,
                'message' => 'User not authenticated'
            ], 401);
        }

        $data = array();

        // 查询当前用户的购物车列表
        $cartItems = FactoryCart::where('user_id', $userId)->orderBy('sort_order')->orderBy('id')->get();

        $useBalance = 0;
        $totalShippingFee = 0;
        $useInventoryArr = [];

        foreach ($cartItems as $key => $cartItem) {
            // 将模型对象转换为数组
            $data[$key] = $cartItem->toArray();

            $useInventoryKey = $cartItem->product_id . '_' . $cartItem->width . '_' . $cartItem->height;

            $inventoryProductTotal = InventoryProduct::findInventoryProductsTotal($data[$key]);

            $data[$key]['inventory_quantity'] = (int)$inventoryProductTotal;

            if (!isset($useInventoryArr[$useInventoryKey])) {
                $useInventoryArr[$useInventoryKey] = 0;
            }

            $data[$key]['use_inventory_quantity'] = min($inventoryProductTotal - $useInventoryArr[$useInventoryKey], $cartItem['quantity']);

            $useInventoryArr[$useInventoryKey] += $data[$key]['use_inventory_quantity'];

            // 计算价格和总计
            if ($cartItem['type'] == 'custom_size') {
                $data[$key]['price'] = max(config('product.min_price'), round($cartItem['price'] * $cartItem['width'] * $cartItem['height'] / 144, 2));
                $data[$key]['total'] = $data[$key]['price'] * $cartItem['quantity'];
            } else {
                $data[$key]['total'] = $cartItem['price'] * $cartItem['quantity'];
            }

            if ($data[$key]['use_inventory_quantity'] < $data[$key]['quantity']) {
                $data[$key]['use_balance_quantity'] = $data[$key]['quantity'] - $data[$key]['use_inventory_quantity'];
                $data[$key]['use_balance'] = $data[$key]['use_balance_quantity'] * $data[$key]['price'];
                $useBalance += $data[$key]['use_balance'];
            } else {
                $data[$key]['use_balance_quantity'] = 0;
                $data[$key]['use_balance'] = 0;
            }

            // 初始化 cart_images 数组
            $data[$key]['cart_images'] = [];

            // 获取并添加相关的图片信息
            $cartImages = CartImage::where('factory_cart_id', $cartItem->id)->get();
            foreach ($cartImages as $cartImage) {
                $data[$key]['cart_images'][] = array(
                    'id' => $cartImage->id, // 使用 $cartImage 的 ID 而不是 $cartItem 的 ID
                    'file_name' => $cartImage->original_name,
                    'preview_url' => Storage::url($cartImage->image_path),
                    'file_size' => $cartImage->file_size,
                    'quantity' => $cartImage->quantity,
                    'rotation' => $cartImage->degree,
                );
            }

            if ($cartItem->shipping_labels) {
                $data[$key]['shipping_label_info'] = $cartItem->shipping_labels;
            } elseif ($cartItem->shipping_label_original_name) {
                $data[$key]['shipping_label_info'][] = [
                    'id' => $cartItem->id,
                    'file_name' => $cartItem->shipping_label_original_name,
                    'preview_url' => Storage::url($cartItem->shipping_label_path),
                    'file_size' => $cartItem->shipping_label_size,
                ];
            } else {
                $data[$key]['shipping_label_info'] = [];
            }

            $barcodes = FactoryOrderBarcode::where('factory_cart_id', $cartItem->id)->get();

            foreach ($barcodes as $barcode) {
                $data[$key][$barcode->type] = array(
                    'id' => $cartItem->id,
                    'file_name' => $barcode->barcode_original_name,
                    'preview_url' => Storage::url($barcode->barcode_path),
                    'file_size' => $barcode->barcode_size,
                );
            }

            $data[$key]['shipping_fee'] = (float)$cartItem->shipping_fee;

            $selectedRate = null;
            if ($cartItem->shipping_service_code && $cartItem->shipping_rates) {
                foreach ($cartItem->shipping_rates as $ri => $rv) {
                    if ($rv['serviceCode'] == $cartItem->shipping_service_code) {
                        $selectedRate = $rv;
                    }
                }
            }
            $data[$key]['selected_rate'] = $selectedRate;

            $mainProduct = MainProduct::findOrFail($cartItem->main_product_id);
            $data[$key]['upload_num'] = $mainProduct ? $mainProduct['upload_num'] : 1;

            $totalShippingFee += $cartItem->shipping_fee;
        }

        // 返回 JSON 响应
        return response()->json([
            'success' => true,
            'data' => $data,
            'use_balance' => $useBalance,
            'total_shipping_fee' => $totalShippingFee
        ]);
    }

    public function removeFromFactoryCart($cart_id)
    {
        // 获取当前已认证用户的 ID
        $userId = Auth::id();

        // 查询指定的购物车项
        $cartItem = FactoryCart::where('id', $cart_id)->where('user_id', $userId)->first();

        // 检查购物车项是否存在
        if (!$cartItem) {
            return response()->json([
                'success' => false,
                'message' => 'Factory cart item not found or does not belong to the user'
            ], 404);
        }

        $sortOrder = $cartItem->sort_order;

        if (!empty($cartItem->shipping_labels[0]['custom_order_sn'])) {
            $shipstationService = new ShipstationService();
            foreach ($cartItem->shipping_labels as $label) {
                $shipstationService->voidLabel($label['custom_order_sn']);
            }
        }

        // 删除购物车项
        $cartItem->delete();

        CartImage::where('factory_cart_id', $cart_id)->where('factory_order_id', 0)->delete();

        FactoryOrderBarcode::where('factory_cart_id', $cart_id)->where('factory_order_id', 0)->delete();

        if (!FactoryCart::where('user_id', $userId)->where('sort_order', $sortOrder)->exists()) {
            FactoryCart::where('user_id', $userId)->where('sort_order', '>', $sortOrder)->update(['sort_order' => DB::raw('sort_order - 1')]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Factory cart item removed successfully'
        ]);
    }

    public function checkFactoryCartQuantity(Request $request,)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:1',
            'cart_id' => 'required|integer'
        ]);

        $user = Auth::user();
        $cartId = $validated['cart_id'];

        try {
            $cartItem = FactoryCart::where('id', $cartId)->where('user_id', $user->id)->first();

            if (!$cartItem) {
                throw new \Exception('Factory cart item not found.');
            }

            $inventoryProductTotal = inventoryProduct::findInventoryProductsTotal($cartItem->toArray());

            $cartProductTotal = FactoryCart::where('user_id', $user->id)
                ->whereNot('id', $cartItem->id)
                ->where('product_id', $cartItem->product_id)
                ->where('width', $cartItem->width)
                ->where('height', $cartItem->height)
                ->sum('quantity');

            $cartProductTotal += $validated['quantity'];

            if ($user->group_id < 10  && !in_array('use_balance', $user->permissions ?? []) && $cartProductTotal > $inventoryProductTotal) {
                throw new \Exception("You do not have sufficient permissions to purchase out of inventory items. Please notify your company's administrator to add inventory.");
            }

            $product = Product::findOrFail($cartItem->product_id);

            if (($product->status != 'In Stock' && $cartProductTotal > $inventoryProductTotal) || $product->quantity < ($cartProductTotal - $inventoryProductTotal)) {
                throw new \Exception($product->main_product_name . ' ' . $product->name . ' is out of stock.');
            }

            return response()->json(['success' => true, 'message' => 'Factory cart quantity is correct.']);
        } catch (\Exception $e) {
            $cartItem = FactoryCart::where('id', $cartId)->where('user_id', $user->id)->first();

            return response()->json([
                'error' => $e->getMessage() ?? 'Failed to update quantity.',
                'message' => $e->getMessage() ?? 'Failed to update quantity.',
                'quantity' => $cartItem->quantity
            ], 500);
        }
    }

    public function updateFactoryCartServiceCode(Request $request, $cartId)
    {
        $validated = $request->validate([
            'service_code' => 'required|string'
        ]);

        $user = Auth::user();

        try {
            $cartItem = FactoryCart::where('id', $cartId)->where('user_id', $user->id)->first();

            if (!$cartItem) {
                throw new \Exception('Factory cart item not found.');
            }

            $shipping_fee = 0;
            if ($cartItem['shipping_rates']) {
                foreach ($cartItem['shipping_rates'] as $key => $val) {
                    if ($val['serviceCode'] == $validated['service_code']) {
                        $shipping_fee = $val['shipmentCost'];
                        break;
                    }
                }
                if ($shipping_fee) {
                    $cartItem->shipping_fee = $shipping_fee;
                }
            }

            $cartItem->shipping_service_code = $validated['service_code'];
            $cartItem->save();

            FactoryCart::where('user_id', $user->id)
                ->where('sort_order', $cartItem->sort_order)
                ->whereNot('id', $cartItem->id)
                ->update([
                    'shipping_service_code' => $validated['service_code']
                ]);

            return response()->json(['success' => true, 'message' => 'Factory cart item updated successfully']);
        } catch (\Exception $e) {
            $cartItem = FactoryCart::where('id', $cartId)->where('user_id', $user->id)->first();

            return response()->json([
                'error' => $e->getMessage() ?? 'Failed to update service_code.',
                'message' => $e->getMessage() ?? 'Failed to update service_code.',
            ], 500);
        }
    }

    public function removeFactoryCartServiceCode(Request $request)
    {
        $factoryCart = FactoryCart::where('id', $request->input('cart_id'))->where('user_id', $request->user()->id)->first();

        $factoryCart->shipping_fee = 0;
        $factoryCart->shipping_rates = null;
        $factoryCart->shipping_shipment = null;
        $factoryCart->shipping_service_code = '';

        $factoryCart->save();

        return response()->json(['success' => true, 'message' => 'Factory cart shipping rate removed successfully']);
    }

    public function updateFactoryCartItem(Request $request, $cartId)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:1'
        ]);

        $user = Auth::user();

        try {
            $cartItem = FactoryCart::where('id', $cartId)->where('user_id', $user->id)->first();

            if (!$cartItem) {
                throw new \Exception('Factory cart item not found.');
            }

            $cartItem->quantity = $validated['quantity'];
            $cartItem->save();

            if (!empty($cartItem->shipping_labels[0]['custom_order_sn'])) {
                $shipstationService = new ShipstationService();

                foreach ($cartItem->shipping_labels as $label) {
                    $shipstationService->voidLabel($label['custom_order_sn']);
                }

                FactoryCart::where('user_id', $cartItem->user_id)->where('sort_order', $cartItem->sort_order)->update(['shipping_label_path' => null, 'shipping_label_original_name' => null, 'shipping_label_size' => 0, 'shipping_labels' => null, 'shipping_fee' => 0, 'shipping_address_key' => null, 'shipping_rates' => null, 'shipping_shipment' => null, 'shipping_service_code' => '']);
            }

            if (FactoryCart::where('user_id', $user->id)->where('sort_order', $cartItem->sort_order)->where('shipping_service_code', '!=', '')->exists()) {
                FactoryCart::where('user_id', $user->id)->where('sort_order', $cartItem->sort_order)->update(['shipping_fee' => 0, 'shipping_rates' => null, 'shipping_shipment' => null, 'shipping_service_code' => '']);
            }

            return response()->json(['success' => true, 'message' => 'Factory cart item updated successfully']);
        } catch (\Exception $e) {
            $cartItem = FactoryCart::where('id', $cartId)->where('user_id', $user->id)->first();

            return response()->json([
                'error' => $e->getMessage() ?? 'Failed to update quantity.',
                'message' => $e->getMessage() ?? 'Failed to update quantity.',
                'quantity' => $cartItem->quantity
            ], 500);
        }
    }

    public function updateFactoryCartImage(Request $request, $cartImageId)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:1'
        ]);

        $cartImage = CartImage::where('id', $cartImageId)->where('user_id', auth()->id())->first();

        if (!$cartImage) {
            return response()->json(['success' => false, 'message' => 'Factory cart image not found'], 404);
        }

        $cartImage->quantity = $validated['quantity'];
        $cartImage->save();

        return response()->json(['success' => true, 'message' => 'Factory cart image updated successfully']);
    }

    public function rotateFactoryCartImage(Request $request, $cartImageId)
    {
        $validated = $request->validate([
            'degree' => 'required|integer'
        ]);

        $cartImage = CartImage::where('id', $cartImageId)->where('user_id', auth()->id())->first();

        if (!$cartImage) {
            return response()->json(['success' => false, 'message' => 'Factory cart image not found'], 404);
        }

        $cartImage->degree = $validated['degree'];
        $cartImage->save();

        return response()->json(['success' => true, 'message' => 'Factory cart image updated successfully']);
    }

    public function updateFactoryCartComments(Request $request)
    {
        $validated = $request->validate([
            'comments' => 'nullable|string',
            'cart_id' => 'required|integer'
        ]);

        $user = Auth::user();

        $cartItem = FactoryCart::where('id', $validated['cart_id'])->where('user_id', $user->id)->first();

        if (!$cartItem) {
            return response()->json(['success' => false, 'message' => 'Factory cart item not found'], 404);
        }

        $cartItem->comments = $validated['comments'];
        $cartItem->save();

        return response()->json(['success' => true, 'message' => 'Factory cart comments updated successfully']);
    }

    public function updateFactoryCartJobId(Request $request)
    {
        $validated = $request->validate([
            'job_id' => 'nullable|string',
            'cart_id' => 'required|integer'
        ]);

        $user = Auth::user();

        $factoryCart = FactoryCart::where('id', $validated['cart_id'])->where('user_id', $user->id)->first();

        if (!$factoryCart) {
            return response()->json(['success' => false, 'message' => 'Factory cart item not found'], 404);
        }

        $factoryCart->job_id = $validated['job_id'];
        $factoryCart->save();

        FactoryCart::where('user_id', $user->id)->where('sort_order', $factoryCart->sort_order)->whereNot('id', $factoryCart->id)->update(['job_id' => $factoryCart->job_id]);

        return response()->json(['success' => true, 'message' => 'Factory cart job ID updated successfully']);
    }

    public function uploadShippingLabel(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:jpeg,png,jpg,pdf|max:204800',
            'file_id' => 'required|exists:factory_carts,id'
        ]);

        try {
            $file = $request->file('file');
            $extension = $file->getClientOriginalExtension();
            $fileName = Str::uuid();
            $manager = new ImageManager(new Driver());

            // Create temp directory if it doesn't exist
            $tempDir = storage_path('app/temp');
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }

            $tempPath = null;

            // Process file based on type
            if (in_array($extension, ['jpeg', 'jpg', 'png'])) {
                // Process image files
                $image = $manager->read($file);

                // Resize image to standard dimensions
                $newWidth = 548;
                $newHeight = 792;
                $image->resize($newWidth, $newHeight);

                // Save processed image to temp file
                $tempPath = storage_path('app/temp/' . Str::uuid() . '.' . $extension);
                $image->save($tempPath);

                $fileName .= '.' . $extension;
            } elseif ($extension === 'pdf') {
                // Check PDF page count
                $pageCount = 0;
                if ($fp = fopen($file->getPathname(), "r")) {
                    $content = fread($fp, filesize($file->getPathname()));
                    $pageCount = preg_match_all("/\/Page\W/", $content, $dummy);
                    fclose($fp);
                }

                if ($pageCount > 1) {
                    throw new \Exception('PDF file cannot contain more than one page.');
                }

                // Convert PDF to image using PdfImageProcessorService
                $tempPath = storage_path('app/temp/' . Str::uuid() . '.pdf');
                $processor = new PdfImageProcessorService();

                // Process the PDF and get the image
                $result = $processor->process($file->getPathname(), $tempPath);

                if ($result === false) {
                    $tempPath = $file->getPathname();
                }

                $fileName .= '.pdf';
            }

            // Create new UploadedFile instance from processed file
            $processedFile = new \Illuminate\Http\UploadedFile(
                $tempPath,
                $file->getClientOriginalName(),
                mime_content_type($tempPath),
                null,
                true
            );

            // Store the processed file
            $path = $processedFile->storeAs('shipping-label', $fileName, 'public');

            $user = Auth::user();
            $factoryCart = FactoryCart::where('id', $request->file_id)
                ->where('user_id', $user->id)
                ->first();

            // Check for duplicate shipping labels
            if (FactoryCart::where('user_id', $user->id)
                ->where('shipping_label_size', $processedFile->getSize())
                ->where('shipping_label_original_name', $file->getClientOriginalName())
                ->whereNot('sort_order', $factoryCart->sort_order)
                ->exists()
            ) {
                throw new \Exception("The shipping label has already been used for another order!");
            }

            // Update factory cart
            $factoryCart->shipping_label_path = $path;
            $factoryCart->shipping_label_original_name = $file->getClientOriginalName();
            $factoryCart->shipping_label_size = $processedFile->getSize();
            $factoryCart->shipping_address_key = md5(file_get_contents($processedFile->getRealPath()));

            $factoryCart->shipping_labels = [[
                'id' => $factoryCart->id,
                'file_name' => $factoryCart->shipping_label_original_name,
                'preview_url' => '/storage/' . $factoryCart->shipping_label_path,
                'file_size' => $factoryCart->shipping_label_size,
            ]];

            $factoryCart->save();

            // Update related factory carts
            FactoryCart::where('user_id', $user->id)
                ->where('sort_order', $factoryCart->sort_order)
                ->whereNot('id', $factoryCart->id)
                ->update([
                    'shipping_label_path' => $factoryCart->shipping_label_path,
                    'shipping_label_original_name' => $factoryCart->shipping_label_original_name,
                    'shipping_label_size' => $factoryCart->shipping_label_size,
                    'shipping_address_key' => $factoryCart->shipping_address_key
                ]);

            // Cleanup temp file
            if ($tempPath && file_exists($tempPath)) {
                unlink($tempPath);
            }

            return response()->json([
                'success' => true,
                'file_url' => Storage::url($path),
                'file_id' => $request->file_id
            ]);
        } catch (\Exception $e) {
            // Ensure temp file cleanup
            if (isset($tempPath) && file_exists($tempPath)) {
                unlink($tempPath);
            }

            return response()->json([
                'success' => false,
                'message' => 'Upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    public function removeShippingLabel(Request $request)
    {
        $factoryCart = FactoryCart::where('id', $request->input('cart_id'))->where('user_id', $request->user()->id)->first();

        if (!empty($factoryCart->shipping_labels[0]['custom_order_sn'])) {
            $shipstationService = new ShipstationService();

            foreach ($factoryCart->shipping_labels as $label) {
                $shipstationService->voidLabel($label['custom_order_sn']);
            }
        }

        FactoryCart::where('user_id', $factoryCart->user_id)->where('sort_order', $factoryCart->sort_order)->update(['shipping_label_path' => null, 'shipping_label_original_name' => null, 'shipping_label_size' => 0, 'shipping_labels' => null, 'shipping_fee' => 0, 'shipping_address_key' => null, 'shipping_rates' => null, 'shipping_shipment' => null]);

        return response()->json(['success' => true, 'message' => 'Factory cart shipping label removed successfully']);
    }

    public function uploadBarcode(Request $request, $type)
    {
        $request->validate([
            'file' => 'required|mimes:jpeg,png,jpg,pdf|max:204800',
            'file_id' => 'required|exists:factory_carts,id',
        ]);

        try {
            $fileName = Str::uuid() . '.' . $request->file('file')->getClientOriginalExtension();
            $path = $request->file('file')->storeAs('factory-order-barcode', $fileName, 'public');

            $barcode = new FactoryOrderBarcode();
            $barcode->factory_cart_id = $request->file_id;
            $barcode->type = $type;
            $barcode->barcode_path = $path;
            $barcode->barcode_original_name = $request->file('file')->getClientOriginalName();
            $barcode->barcode_size = $request->file('file')->getSize();
            $barcode->save();

            return response()->json([
                'success' => true,
                'file_url' => Storage::url($path),
                'file_id' => $request->file_id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    public function removeBarcode(Request $request)
    {
        $request->validate([
            'factory_cart_id' => 'required|exists:factory_carts,id',
            'type' => 'required|string'
        ]);

        $barcode = FactoryOrderBarcode::where('factory_cart_id', $request->input('factory_cart_id'))->where('type', $request->input('type'))->first();
        $barcode->delete();

        return response()->json(['success' => true, 'message' => $request->input('type') . ' barcode removed successfully']);
    }

    public function duplicateFactoryCart(Request $request)
    {
        try {
            $userId = Auth::id();
            if (!$userId) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            $cartId = $request->input('cart_id');
            $cart = FactoryCart::findOrFail($cartId);
            $validatedData = $cart->toArray();

            // Check inventory availability
            $inventoryProductTotal = InventoryProduct::findInventoryProductsTotal($validatedData);
            $cartProductTotal = FactoryCart::where('user_id', $userId)
                ->where('product_id', $validatedData['product_id'])
                ->where('width', $validatedData['width'])
                ->where('height', $validatedData['height'])
                ->sum('quantity');

            /*if($inventoryProductTotal < $cartProductTotal) {
                return response()->json([
                    'success' => false,
                    'message' => "Insufficient stock for product {$validatedData['main_product_name']} {$validatedData['product_name']}: requested {$cartProductTotal}, available {$inventoryProductTotal}"
                ], 400);
            }*/

            if ($inventoryProductTotal < $cartProductTotal + $validatedData['quantity']) {
                $useInventoryQty = max(0, $inventoryProductTotal - $cartProductTotal);
                $useBalanceQty = $validatedData['quantity'] - $useInventoryQty;
            } else {
                $useInventoryQty = $validatedData['quantity'];
                $useBalanceQty = 0;
            }

            DB::beginTransaction();

            $newCart = $cart->replicate();
            $newCart->user_id = $userId;
            if (!$request->duplicate_address) {
                $newCart->shipping_label_path = '';
                $newCart->shipping_label_original_name = NULL;
                $newCart->shipping_label_size = 0;
            }

            $newCart->save();

            $data = $newCart->toArray();

            if (!$request->duplicate_address) {
                $cartImages = CartImage::where('factory_cart_id', $cartId)->get();
                foreach ($cartImages as $image) {
                    $newImage = $image->replicate();
                    $newImage->factory_cart_id = $newCart->id;
                    $newImage->save();
                }
            } else {
                $barcodes = FactoryOrderBarcode::where('factory_cart_id', $cartId)->get();

                foreach ($barcodes as $barcode) {
                    $newBarcode = $barcode->replicate();
                    $newBarcode->factory_cart_id = $newCart->id;
                    $newBarcode->save();

                    $data[$barcode->type] = array(
                        'id' => $newCart->id,
                        'file_name' => $barcode->barcode_original_name,
                        'preview_url' => Storage::url($barcode->barcode_path),
                        'file_size' => $barcode->barcode_size,
                    );
                }
            }

            DB::commit();

            if ($data['type'] == 'custom_size') {
                $data['price'] = max(config('product.min_price'), round($data['price'] * $data['width'] * $data['height'] / 144, 2));
                $data['total'] = $data['price'] * $data['quantity'];
            } else {
                $data['total'] = $data['price'] * $data['quantity'];
            }

            $data['use_inventory_quantity'] = $useInventoryQty;
            $data['use_balance_quantity'] = $useBalanceQty;
            $data['use_balance'] = $data['price'] * $data['use_balance_quantity'];

            $data['cart_images'] = [];
            $newCartImages = CartImage::where('factory_cart_id', $newCart->id)->get();
            foreach ($newCartImages as $cartImage) {
                $data['cart_images'][] = [
                    'id' => $cartImage->id,
                    'file_name' => $cartImage->original_name,
                    'preview_url' => Storage::url($cartImage->image_path),
                    'file_size' => $cartImage->file_size,
                    'quantity' => $cartImage->quantity,
                    'rotation' => $cartImage->degree,
                ];
            }

            $data['shipping_label_info'] = $newCart->shipping_label_original_name ? [
                'id' => $newCart->id,
                'file_name' => $newCart->shipping_label_original_name,
                'preview_url' => Storage::url($newCart->shipping_label_path),
                'file_size' => $newCart->shipping_label_size,
            ] : [];

            return response()->json([
                'success' => true,
                'cart' => $data
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function getCompanyBalance()
    {
        $user = Auth::user();

        $companyAdmin = User::where('company_id', $user->company_id)->whereIn('group_id', [10, 99])->where('status', 'Active')->first();

        return response()->json([
            'balance' => $companyAdmin->balance ?? 0
        ]);
    }

    public function addNewItem(Request $request)
    {
        // 验证请求的数据
        $validatedData = $request->validate([
            'sort_order' => 'required|integer',
            'product_id' => 'required|integer',
            'width' => 'nullable|numeric',
            'height' => 'nullable|numeric',
            'option' => 'nullable|json',
        ]);

        $user = Auth::user();
        $userId = $user->id;

        $product = Product::findOrFail($validatedData['product_id']);

        $mainProduct = MainProduct::findOrFail($product->main_product_id);

        $jsonService = new JsonService();

        $inventoryProductTotal = inventoryProduct::findInventoryProductsTotal($validatedData);

        $cartProductTotal = FactoryCart::where('user_id', $userId)
            ->where('product_id', $validatedData['product_id'])
            ->where('width', $validatedData['width'])
            ->where('height', $validatedData['height'])
            ->sum('quantity');

        $cartProductTotal += 1;

        /*if($inventoryProductTotal <  $cartProductTotal) {
            return response()->json(['error' => "Insufficient stock for product {$validatedData['main_product_name']} {$validatedData['product_name']}: requested {$cartProductTotal}, available {$inventoryProductTotal}"], 500);
        }*/

        $jsonService = new JsonService();

        $validatedData['option'] = $jsonService->normalizeJson($validatedData['option']);

        try {
            $product = Product::findOrFail($validatedData['product_id']);

            if ($product->status != 'In Stock' || $product->quantity < ($cartProductTotal - $inventoryProductTotal)) {
                throw new \Exception($product->main_product_name . ' ' . $product->name . ' is out of stock.');
            }

            $sortOrder = $validatedData['sort_order'] ? $validatedData['sort_order'] : (FactoryCart::where('user_id', $userId)->max('sort_order') ?? 0) + 1;

            $firstFactoryCart = FactoryCart::where('sort_order', $sortOrder)->where('user_id', $userId)->first();

            if (!empty($firstFactoryCart->shipping_labels[0]['custom_order_sn'])) {
                $shipstationService = new ShipstationService();

                foreach ($firstFactoryCart->shipping_labels as $label) {
                    $shipstationService->voidLabel($label['custom_order_sn']);
                }
                FactoryCart::where('user_id', $firstFactoryCart->user_id)->where('sort_order', $firstFactoryCart->sort_order)->update(['shipping_label_path' => null, 'shipping_label_original_name' => null, 'shipping_label_size' => 0, 'shipping_labels' => null, 'shipping_fee' => 0, 'shipping_address_key' => null, 'shipping_rates' => null, 'shipping_shipment' => null, 'shipping_service_code' => '']);
            }

            if ($firstFactoryCart) {
                if (FactoryCart::where('user_id', $user->id)->where('sort_order', $firstFactoryCart->sort_order)->where('shipping_service_code', '!=', '')->exists()) {
                    FactoryCart::where('user_id', $user->id)->where('sort_order', $firstFactoryCart->sort_order)->update(['shipping_fee' => 0, 'shipping_rates' => null, 'shipping_shipment' => null, 'shipping_service_code' => '']);
                }
            }

            $newFactoryCart = FactoryCart::create([
                'user_id' => $userId,
                'main_product_id' => $mainProduct->id,
                'main_product_name' => $mainProduct->name,
                'product_id' => $product->id,
                'product_name' => $product->name,
                'product_image' => $mainProduct->image,
                'quantity' => 1,
                'width' => $validatedData['width'],
                'height' => $validatedData['height'],
                'price' => $product->price,
                'type' => $product->type,
                'option' => $validatedData['option'],
                'sort_order' => $sortOrder,
                'shipping_label_path' => $firstFactoryCart->shipping_label_path ?? null,
                'shipping_label_original_name' => $firstFactoryCart->shipping_label_original_name ?? null,
                'shipping_label_size' => $firstFactoryCart->shipping_label_size ?? 0,
                'job_id' => $firstFactoryCart->job_id ?? null,
            ]);

            $insufficientStock = false;

            if ($inventoryProductTotal <  $cartProductTotal) {
                $insufficientStock = "Insufficient stock for product {$mainProduct->name} {$product->name}: requested {$cartProductTotal}, available {$inventoryProductTotal}, insufficient stock will be purchased using the balance.\n";
            }

            return response()->json([
                'success' => true,
                'message' => 'Product sent to print successfully!',
                'insufficient_stock' => $insufficientStock
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage() ?? 'Failed to send product to print.',
                'message' => $e->getMessage() ?? 'Failed to send product to print.'
            ], 500);
        }
    }

    public function getShippingRates(Request $request, $id)
    {
        $shipstationService = new ShipstationService();


        try {
            $response = $shipstationService->sendShippingRequest($request, $id);
            return response()->json($response);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    public function getBestRate(Request $request, $id)
    {
        try {
            $shipstationService = new ShipstationService();

            $response = $shipstationService->getRateList($request, $id);
            return response()->json($response);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }
}
