<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use App\Models\Order;
use App\Models\Product;
use App\Models\MainProduct;
use App\Models\FactoryOrder;
use App\Models\FactoryCart;
use App\Models\InventoryProduct;
use App\Models\CartImage;
use App\Models\Company;
use App\Models\User;
use App\Models\FactoryOrderBarcode;
use DB;
use App\Services\EmailService;
use App\Services\ImageBlurEdgeService;
use App\Services\ImageMirrorService;
use App\Services\FactoryService;
use App\Services\LogisticsService;
use App\Services\V2\ShipstationService;
use App\Services\MingtengDeliveryService;
use Carbon\Carbon;

class FactoryOrderController extends Controller
{
    public function addFactoryOrder(Request $request)
    {
        try {
            $orderItems = [];
            $factoryOrder = [];
            $emailData = [];

            DB::transaction(function () use (&$orderItems, &$factoryOrder, &$emailData) {
                $user = Auth::user();

                // 锁定当前用户的所有购物车条目
                $carts = FactoryCart::where('user_id', $user->id)->lockForUpdate()->orderBy('sort_order')->orderBy('shipping_label_size', 'DESC')->get();
                
                if ($carts->isEmpty()) {
                    throw new \Exception('Cart is empty.');
                }

                $useBalance = 0;
                $totalShippingFee = 0;
                $data = array();
                $useInventoryArr = [];
                $cartGroup = [];

                // 首先验证所有商品
                foreach ($carts as $key => $cart) {
                    $data[$key] = $cart->toArray();

                    $cartImagesTotal = CartImage::where('factory_cart_id', $cart->id)->sum('quantity');

                    $mainProduct = MainProduct::findOrFail($cart->main_product_id);
                    $upload_num = $mainProduct ? $mainProduct['upload_num'] : 1;

                    if ($cartImagesTotal != $cart->quantity * $upload_num) {
                        throw new \Exception("(" . $cart->sort_order .  ") Please ensure that the number of printed images matches the number of items! (" . $cartImagesTotal . " images don't match " . $cart->quantity * $cart->upload_num . " items)");
                    }

                    $useInventoryKey = $cart->product_id . '_' . $cart->width . '_' . $cart->height;

                    $inventoryProductTotal = InventoryProduct::findInventoryProductsTotal($data[$key]);

                    if(!isset($useInventoryArr[$useInventoryKey])) {
                        $useInventoryArr[$useInventoryKey] = 0;
                    }

                    $data[$key]['use_inventory_quantity'] = min($inventoryProductTotal - $useInventoryArr[$useInventoryKey], $cart->quantity);

                    $useInventoryArr[$useInventoryKey] += $data[$key]['use_inventory_quantity'];

                    // 计算价格和总计
                    if($cart->type == 'custom_size') {
                        $cart->item_price = max(config('product.min_price'), round($cart->price * $cart->width * $cart->height / 144, 2));
                    } else {
                        $cart->item_price = $cart->price;
                    }

                    $cart->total = $cart->item_price * $cart->quantity;

                    if ($data[$key]['use_inventory_quantity'] < $cart->quantity) {
                        $cart->use_balance_quantity = $cart->quantity - $data[$key]['use_inventory_quantity'];
                        $cart->use_balance = $cart->use_balance_quantity * $cart->item_price;
                        $useBalance += $cart->use_balance;
                    } else {
                        $cart->use_balance_quantity = 0;
                        $cart->use_balance = 0;
                    }

                    if(empty($cart->shipping_address_key) && empty($cart->shipping_service_code)) {
                        throw new \Exception("Please upload all shipping label or select the shipping rates!");
                    }

                    $totalShippingFee += $cart->shipping_fee;

                    $orderItems[] = $cart;
                    
                    $cartGroup[$cart->sort_order][] = $cart;
                }

                if($user->group_id < 10 && !in_array('use_balance', $user->permissions) && $useBalance) {
                    throw new \Exception('You do not have sufficient permissions to purchase out-of-stock items. Please notify your company’s administrator to add inventory.');
                }

                if($useBalance || $totalShippingFee) {
                    $companyAdmin = User::where('company_id', $user->company_id)->whereIn('group_id', [10, 99])->where('status', 'Active')->first();

                    if($useBalance + $totalShippingFee > $companyAdmin->balance) {
                        throw new \Exception("Your balance is insufficient!");
                    } else {
                        $companyAdmin->balance -= ($useBalance + $totalShippingFee);
                        $companyAdmin->save();

                        $purchaseOrder = Order::create([
                            'user_id' => $user->id,
                            'user_email' => $user->email,
                            'company_id' => $user->company_id, 
                            'total' => $useBalance,
                            'use_balance' => $useBalance,
                            'payment_method' => 'balance',
                            'payment_information' => 'Paying for insufficient stock with balance',
                            'status' => 'Completed'
                        ]);

                        // 所有验证通过后，再进行库存扣减
                        foreach ($orderItems as $cart) {
                            if($cart->use_balance_quantity) {
                                $product = Product::lockForUpdate()->find($cart->product_id);
                                $product->quantity -= $cart->use_balance_quantity;
                                $product->save();

                                InventoryProduct::create([
                                    'order_id' => $purchaseOrder->id,
                                    'user_id' => $user->id, 
                                    'company_id' => $user->company_id, 
                                    'main_product_id' => $product->main_product_id,
                                    'main_product_name' => $product->main_product_name,
                                    'product_id' => $product->id,
                                    'product_name' => $product->name,
                                    'product_image' => $cart->product_image,
                                    'width' => $cart->width,
                                    'height' => $cart->height,
                                    'price' => $cart->price,
                                    'type' => $cart->type,
                                    'option' => $cart->option,
                                    'inventory_type' => 'in',
                                    'quantity' => $cart->use_balance_quantity,
                                    'job_id' => $cart->job_id,
                                    'comments' => $cart->comments,
                                ]);
                            }
                        }
                    }
                }

                foreach($cartGroup as $o => $val) {
                    if($val[0]->shipping_service_code) {
                        //创建运单
                        $shipstationService = new ShipstationService();
                        $shipstationService->createShippingLabel($val);
                    }
                }
                //获取新购物车内容
                $newCarts = FactoryCart::where('user_id', $user->id)->orderBy('sort_order')->orderBy('shipping_label_size', 'DESC')->get();

                $orderNewItems = [];
                $newData = [];
                foreach($newCarts as $cart) {
                    $newData[$key] = $cart->toArray();

                    $useInventoryKey = $cart->product_id . '_' . $cart->width . '_' . $cart->height;

                    $inventoryProductTotal = InventoryProduct::findInventoryProductsTotal($newData[$key]);

                    if(!isset($useInventoryArr[$useInventoryKey])) {
                        $useInventoryArr[$useInventoryKey] = 0;
                    }

                    $newData[$key]['use_inventory_quantity'] = min($inventoryProductTotal - $useInventoryArr[$useInventoryKey], $cart->quantity);

                    $useInventoryArr[$useInventoryKey] += $newData[$key]['use_inventory_quantity'];

                     // 计算价格和总计
                    if($cart->type == 'custom_size') {
                        $cart->item_price = max(config('product.min_price'), round($cart->price * $cart->width * $cart->height / 144, 2));
                    } else {
                        $cart->item_price = $cart->price;
                    }

                    $cart->total = $cart->item_price * $cart->quantity;

                    if ($newData[$key]['use_inventory_quantity'] < $cart->quantity) {
                        $cart->use_balance_quantity = $cart->quantity - $newData[$key]['use_inventory_quantity'];
                        $cart->use_balance = $cart->use_balance_quantity * $cart->item_price;
                        $useBalance += $cart->use_balance;
                    } else {
                        $cart->use_balance_quantity = 0;
                        $cart->use_balance = 0;
                    }

                    $orderNewItems[] = $cart;
                }

                $orders = [];
                $totalAmount = [];
                $totalQuantity = [];
                $totalUseBalane = [];
                $totalOrderShippingFee = [];
                $jobIds = [];
                $shippingLabels = [];

                foreach ($orderNewItems as $item) {
                    $orders[$item->shipping_address_key][] = $item;

                    if(!isset($totalAmount[$item->shipping_address_key])) {
                        $totalAmount[$item->shipping_address_key] = 0;
                    }

                    if(!isset($totalQuantity[$item->shipping_address_key])) {
                        $totalQuantity[$item->shipping_address_key] = 0;
                    }

                    if(!isset($totalUseBalane[$item->shipping_address_key])) {
                        $totalUseBalane[$item->shipping_address_key] = 0;
                    }

                    if(!isset($totalOrderShippingFee[$item->shipping_address_key])) {
                        $totalOrderShippingFee[$item->shipping_address_key] = 0;
                    }

                    $totalAmount[$item->shipping_address_key] += $item->total;
                    $totalQuantity[$item->shipping_address_key] += $item->quantity;
                    $totalUseBalane[$item->shipping_address_key] += $item->use_balance;
                    $totalOrderShippingFee[$item->shipping_address_key] += $item->shipping_fee;

                    if($item->job_id) {
                        $jobIds[$item->shipping_address_key][] = $item->job_id;
                    }

                    if (!isset($shippingLabels[$item->shipping_address_key])) {
                        $shippingLabels[$item->shipping_address_key] = [];
                    }

                    if($item->shipping_labels) {
                        $shippingLabels[$item->shipping_address_key] = [...$shippingLabels[$item->shipping_address_key], ...$item->shipping_labels];
                    }
                }

                $products = '';
                $date = new \DateTime();
                $timestamp = $date->format('YmdHisv');
                $printJobId = $timestamp . '-' . str_pad($user->id, 8, '0', STR_PAD_LEFT);
                $firstOrderId = '';
                $lastOrderId = '';

                foreach($orders as $key => $order) {
                    $factoryOrder = FactoryOrder::create([
                        'print_job_id' => $printJobId,
                        'user_id' => $user->id, 
                        'user_email' => $user->email,
                        'company_id' => $user->company_id, 
                        'total' => $totalAmount[$key],
                        'use_balance' => $totalUseBalane[$key],
                        'total_shipping_fee' => $totalOrderShippingFee[$key],
                        'quantity' => $totalQuantity[$key],
                        'shipping_label' => $order[0]->shipping_label_path,
                        'shipping_labels' => $shippingLabels[$key],
                        'user_job_id' => empty($jobIds[$key]) ? '' : implode(',' , $jobIds[$key]),
                        'status' => 'Pending',
                    ]);

                    $products .= "Print Order #{$factoryOrder->id} ({$totalQuantity[$key]} items)\n";
                    $comments = '';

                    foreach($order as $orderCart) {
                        CartImage::where('factory_cart_id', $orderCart->id)->update(['factory_order_id' => $factoryOrder->id, 'job_id' => $orderCart->job_id, 'comments' => $orderCart->comments]);
                        FactoryOrderBarcode::where('factory_cart_id', $orderCart->id)->update(['factory_order_id' => $factoryOrder->id]);

                        InventoryProduct::create([
                            'order_id' => $factoryOrder->id,
                            'user_id' => $user->id, 
                            'company_id' => $user->company_id, 
                            'main_product_id' => $orderCart->main_product_id,
                            'main_product_name' => $orderCart->main_product_name,
                            'product_id' => $orderCart->product_id,
                            'product_name' => $orderCart->product_name,
                            'product_image' => $orderCart->product_image,
                            'width' => $orderCart->width,
                            'height' => $orderCart->height,
                            'price' => $orderCart->price,
                            'type' => $orderCart->type,
                            'option' => $orderCart->option,
                            'inventory_type' => 'out',
                            'quantity' => -$orderCart->quantity,
                            'job_id' => $orderCart->job_id,
                            'comments' => $orderCart->comments,
                        ]);

                        if($orderCart->comments) {
                            $comments .= $orderCart->comments . "\n";
                        }

                        $products .= $orderCart->main_product_name . ' ' . $this->formatSize($orderCart->width) . '" x ' . $this->formatSize($orderCart->height)  . '"  qty:' . $orderCart->quantity . "\n";         
                    }

                    $factoryOrder->comments = $comments;
                    $factoryOrder->status = 'Ready To Proceed';
                    $factoryOrder->save();

                    $factoryService = new FactoryService();
                    $factoryService->sendOrder($factoryOrder->id);

                    if (!$firstOrderId) {
                       $firstOrderId = $factoryOrder->id;
                   }
                   $lastOrderId = $factoryOrder->id;
                }

                $emailData['id'] = $firstOrderId != $lastOrderId ? $firstOrderId . ' - ' . $lastOrderId : $lastOrderId;
                $emailData['user_email'] = $user->email;
                $emailData['products'] = $products;
                
                // 清空购物车
                FactoryCart::where('user_id', $user->id)->delete();
            });
            
            $emailService = new EmailService();
            $response = $emailService->sendNotification($emailData, 'inventory');
            
            return response()->json([
                'success' => true,
                'message' => 'Order created successfully',
                'order' => $factoryOrder,
            ], 201);
            
        } catch (\Exception $e) {
            \Log::error('Checkout failed: ' . $e->getMessage(), [
                'user_id' => Auth::id(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'error' => $e->getMessage(),
                'message' => $e->getMessage() ?? 'Checkout failed. Please try again.'
            ], 400);
        }
    }

    public function formatSize($number) {
        if (fmod($number, 1) === 0.00) {
            return (int) $number;
        } else {
            return $number;
        }
    }

    public function getFactoryOrderDetails($id)
    {
        try {
            $user = Auth::user();

            // 获取订单信息，同时加载关联的图片信息
            $order = FactoryOrder::with(['cartImages' => function($query) {
                $query->select([
                    'id',
                    'factory_order_id',
                    'image_path',
                    'original_name',
                    'main_product_name',
                    'product_name',
                    'quantity',
                    'width',
                    'height',
                    'option',
                    'job_id'
                ]);
            }])->where('company_id', $user->company_id)->findOrFail($id);

            $company = company::find($user->company_id);

            foreach($order->cartImages as $key => $cartImage) {
                $thumbnail = str_replace(['cart-images', 'company-gallery'], ['cart-thumbnails', 'company-thumbnails'], $cartImage->image_path);
                $largeThumbnail = str_replace(['cart-images', 'company-gallery'], ['large-cart-thumbnails', 'large-company-thumbnails'], $cartImage->image_path);
                if (file_exists(Storage::path('public/' . $thumbnail))) {
                    $cartImage->image_path = $thumbnail;
                    if (Storage::exists('public/' . $largeThumbnail)) {
                        $order->cartImages[$key]->large_thumbnail = $largeThumbnail;
                    }
                }
            }

            $carbon = Carbon::createFromFormat('Y-m-d H:i:s', $order->created_at);
            $carbon->addMinutes(10);

            return response()->json([
                'success' => true,  
                'order' => [
                    'id' => $order->id,
                    'print_job_id' => $order->print_job_id,
                    'company' => $company->name,
                    'user_email' => $order->user_email,
                    'user_job_id' => $order->user_job_id,
                    'shipping_label' => $order->shipping_label,
                    'shipping_labels' => $order->shipping_labels,
                    'total_shipping_fee' => $order->total_shipping_fee,
                    'sku_barcode' => $barcodeUrls['sku_barcode'] ?? '',
                    'package_barcode' => $barcodeUrls['package_barcode'] ?? '',
                    'total' => $order->total,
                    'quantity' => $order->quantity,
                    'use_balance' => $order->use_balance,
                    'status' => $order->status,
                    'created_at' => Carbon::parse($order->created_at)->toDateTimeString(),
                    'cancel_timestamp' => $carbon->timestamp,
                ],
                'images' => $order->cartImages
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to Retrieve Order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function cancelFactoryOrder($id): JsonResponse
    {
        try {
            $user = Auth::user();

            DB::beginTransaction();

            if($user->group_id == 99) {
                $factoryOrder = FactoryOrder::findOrFail($id);
                $user = User::find($factoryOrder->user_id);
            } else {
                $factoryOrder = FactoryOrder::where('id', $id)->where('company_id', $user->company_id)->first();
            }

            if (!$factoryOrder) {
                throw new \Exception('Order not found.');
            }

            InventoryProduct::where('order_id', $id)->where('inventory_type', 'out')->where('quantity', '<', 0)->update([
                'status' => 'Canceled',
                'quantity' => 0
            ]);

            $shippingLabelSns = [];
            $deliveryService = '';

            foreach($factoryOrder->shipping_labels as $shippingLabel) {
                if(!empty($shippingLabel['custom_order_sn'])) {
                    $shippingLabelSns[] = $shippingLabel['custom_order_sn'];
                }
            }

            if($shippingLabelSns) {
                $shipstationService = new ShipstationService();

                foreach(array_unique($shippingLabelSns) as $shippingLabelSn) {
                    $shipstationService->voidLabel($shippingLabelSn);
                }
            } 

            if($user->company_id == $factoryOrder->company_id && ($factoryOrder->use_balance > 0 || $factoryOrder->total_shipping_fee > 0)) {
                $companyAdmin = User::where('company_id', $user->company_id)->where('group_id', '>=', 10)->first();
                if (!$companyAdmin) {
                    $companyAdmin = User::where('id', $user->id)->first();
                }
                $companyAdmin->balance += $factoryOrder->use_balance + $factoryOrder->total_shipping_fee;
                $companyAdmin->save();
            }

            $factoryOrder->status = 'Canceled';
            $factoryOrder->save();

            DB::commit();

            $factoryService = new FactoryService();
            $ret = $factoryService->changeOrder($factoryOrder->id, $factoryOrder->status, 7);

            return response()->json([
                'success' => true,  
                'status' => $factoryOrder->status,
            ]);
               

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to Retrieve Order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function updateShippingLabel(Request $request, $id)
    {
        $request->validate([
            'shipping_label.*' => 'required|file|mimes:jpeg,png,jpg,gif,pdf|max:10240' // 10MB max per file
        ]);

        try {
            $user = Auth::user();
            $factoryOrder = FactoryOrder::where('id', $id)
                ->where('company_id', $user->company_id)
                ->first();

            if (!$factoryOrder) {
                throw new \Exception('Order not found.');
            }

            if ($factoryOrder->status != 'Ready To Proceed') {
                throw new \Exception('Order status does not allow shipping label update.');
            }

            $uploadedFiles = [];
            
            foreach ($request->file('shipping_label') as $file) {
                $fileName = Str::uuid() . '.' . $file->getClientOriginalExtension();
                $path = $file->storeAs('shipping-label', $fileName, 'public');
                
                $uploadedFiles[] = [
                    'preview_url' => '/storage/' . $path,
                    'file_name' => $file->getClientOriginalName(),
                    'file_size' => $file->getSize(),
                ];
            }

            // 如果你的数据库字段是 JSON 类型，可以直接存储数组
            $factoryOrder->shipping_labels = $uploadedFiles;
            $factoryOrder->save();

            return response()->json([
                'success' => true,
                'shipping_labels' => $uploadedFiles
            ]);

        } catch (\Exception $e) {
            // 如果上传过程中出错，删除已上传的文件
            if (isset($uploadedFiles)) {
                foreach ($uploadedFiles as $file) {
                    Storage::disk('public')->delete($file['path']);
                }
            }

            return response()->json([
                'success' => false,
                'message' => 'Upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateUserJobId(Request $request, $id)
    {
        $request->validate([
            'user_job_id' => 'required|string' // 10MB max
        ]);

        try {
            $user = Auth::user();

            $factoryOrder = FactoryOrder::where('id', $id)->where('company_id', $user->company_id)->first();
            $factoryOrder->user_job_id = $request->user_job_id;
            $factoryOrder->save();

            return response()->json([
                'success' => true,
                'user_job_id' => $factoryOrder->user_job_id,
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    private function ensureJpgExtension(string $path): string
    {
        $pathInfo = pathinfo($path);
        
        // 无论输入路径是什么，都确保输出为.jpg
        return ($pathInfo['dirname'] !== '.' ? $pathInfo['dirname'] . DIRECTORY_SEPARATOR : '') 
            . $pathInfo['filename'] 
            . '.jpg';
    }
}