<?php

namespace App\Http\Controllers\Inventory;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\MainProduct;
use App\Models\InventoryProduct;
use App\Models\FactoryCart;
use App\Models\FilterInventoryProduct;
use App\Models\User;
use App\Models\InventoryTransferHistory;
use App\Services\JsonService;
use DB;

class InventoryProductController extends Controller
{
    /**
     * 获取所有状态为 active 的主产品及其关联的产品
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getInventoryProducts()
    {
        $data = [];

        $user = Auth::user();

        $mainProductIds = InventoryProduct::distinct()->where('company_id', $user->company_id)->pluck('main_product_id')->toArray();
        $mainProducts = MainProduct::whereIn('id', $mainProductIds)->get();

        foreach ($mainProducts as $key => $mainProduct) {
            $data[$key]['show'] = false;
            $products = InventoryProduct::getGroupedData($mainProduct->id);
            $data[$key] = $mainProduct->toArray();
            foreach($products as $index => $product) {
                $productData = $product->toArray();
                $productData['index'] = $index + 1;
                $cartQuantity = FactoryCart::findCartInventoryProductsTotal(Auth::id(), $productData);
                $productData['quantity'] -= $cartQuantity;

                if($productData['quantity'] < 0) {
                    $productData['quantity'] = 0;
                }
                
                $filter = FilterInventoryProduct::where('user_id', $user->id)->where('product_id', $product->id)->where('width', $product->width)->where('height', $product->height)->first();

                if(!empty($filter->id)) {
                    $productData['show'] = $filter->show;
                } else {
                    FilterInventoryProduct::create([
                        'user_id' => $user->id, 
                        'main_product_id' => $mainProduct->id,
                        'product_id' => $product->id,
                        'width' => $product->width,
                        'height' => $product->height, 
                        'show' => true,
                    ]);

                    $productData['show'] = true;
                }

                $productData['name'] = $this->formatNumber($productData['width']) . '" x ' .  $this->formatNumber($productData['height']) . '"';

                if($user->group_id > 9 || $productData['quantity'] > 0 || in_array('use_balance', $user->permissions)) {
                    $data[$key]['products'][] = $productData;
                }

                if($productData['show']) {
                    $data[$key]['show'] = true;
                }
            }
        }

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    public function filterInventoryProduct(Request $request) {
        $validatedData = $request->validate([
            'product_id' => 'required|integer',
            'width' => 'required|numeric',
            'height' => 'required|numeric',
            'show' => 'required|boolean',
        ]);

        $user = $request->user();

        $filter = FilterInventoryProduct::where('user_id', $user->id)->where('product_id', $request->product_id)->where('width', $request->width)->where('height', $request->height)->first();
        $filter->show = $request->show;
        $filter->save();

        return response()->json([
            'success' => true,
            'data' => $filter
        ]);
    }

    public function formatNumber($number) {
        return ($number == floor($number)) ? intval($number) : $number;
    }

    public function transferInventoryProduct(Request $request) {
        $validatedData = $request->validate([
            'product_id' => 'required|integer',
            'width' => 'required|numeric',
            'height' => 'required|numeric',
            'user_email' => 'required|email|exists:users,email',
            'quantity' => 'required|integer|min:1',
        ]);

        $user = $request->user();

        if($user->group_id < 10) {
            throw new \Exception("You do not have sufficient permissions to perform this operation!");
        }

        $receiveUser = User::where('email', $validatedData['user_email'])->with('company')->first();

        if($receiveUser->company_id == $user->company_id) {
            throw new \Exception("Users within the same company are not allowed to transfer inventory products!");
        }

        DB::transaction(function () use ($validatedData, $user, $receiveUser) {
            $total = InventoryProduct::findInventoryProductsTotal($validatedData);
            if($validatedData['quantity'] > $total) {
                throw new \Exception("You do not have enough inventory!");
            }

            $inventoryProduct = InventoryProduct::where('company_id', $user->company_id)->where('product_id', $validatedData['product_id'])->where('width', $validatedData['width'])->where('height', $validatedData['height'])->where('quantity', '>', 0)->orderBy('id', 'DESC')->first();

            $record = InventoryTransferHistory::create([
                'main_product_name' => $inventoryProduct->main_product_name,
                'product_id' => $inventoryProduct->product_id,
                'product_name' => $inventoryProduct->product_name,
                'width' => $inventoryProduct->width,
                'height' => $inventoryProduct->height,
                'quantity' => $validatedData['quantity'],
                'from_user_id' => $user->id,
                'to_user_id' => $receiveUser->id,
                'to_user_name' => $receiveUser->name,
                'to_user_email' => $receiveUser->email,
                'to_user_company' => $receiveUser->company->name,
            ]);

            InventoryProduct::create([
                'order_id' => $record->id,
                'user_id' => $user->id, 
                'company_id' => $user->company_id, 
                'main_product_id' => $inventoryProduct->main_product_id,
                'main_product_name' => $inventoryProduct->main_product_name,
                'product_id' => $inventoryProduct->product_id,
                'product_name' => $inventoryProduct->product_name,
                'product_image' => $inventoryProduct->product_image,
                'width' => $inventoryProduct->width,
                'height' => $inventoryProduct->height,
                'price' => $inventoryProduct->price,
                'type' => $inventoryProduct->type,
                'option' => $inventoryProduct->option,
                'inventory_type' => 'out',
                'quantity' => -$validatedData['quantity'],
                'job_id' => '',
                'comments' => 'Transfer inventory to user ID: ' . $receiveUser->id,
            ]);

            InventoryProduct::create([
                'order_id' => $record->id,
                'user_id' => $receiveUser->id, 
                'company_id' => $receiveUser->company_id, 
                'main_product_id' => $inventoryProduct->main_product_id,
                'main_product_name' => $inventoryProduct->main_product_name,
                'product_id' => $inventoryProduct->product_id,
                'product_name' => $inventoryProduct->product_name,
                'product_image' => $inventoryProduct->product_image,
                'width' => $inventoryProduct->width,
                'height' => $inventoryProduct->height,
                'price' => $inventoryProduct->price,
                'type' => $inventoryProduct->type,
                'option' => $inventoryProduct->option,
                'inventory_type' => 'in',
                'quantity' => $validatedData['quantity'],
                'job_id' => '',
                'comments' => 'Received inventory from user ID: ' . $user->id,
            ]);
            
        });

        return response()->json([
            'success' => true,
        ]);
    }

    public function getTransferInventoryRecords(Request $request) {
        // 获取查询参数
        $searchQuery = $request->input('q', '');
        $page = $request->input('page', 1);
        $itemsPerPage = $request->input('itemsPerPage', 10);
        $sortBy = $request->input('sortBy', 'created_at');
        $orderBy = $request->input('orderBy', 'desc');

        $user = $request->user();

        // 查询订单数据
        $query = InventoryTransferHistory::query()->where('from_user_id', $user->id);

        // 如果有搜索查询，则按条件过滤（根据实际需要修改字段）
        if ($searchQuery) {
            $query->where('user_email', 'like', '%' . $searchQuery . '%')
                  ->orWhere('id', 'like', '%' . $searchQuery . '%')
                  ->orWhere('status', 'like', '%' . $searchQuery . '%')
                  ->orWhere('created_at', 'like', '%' . $searchQuery . '%')
                  ->orWhere('total', $searchQuery);
        }

        // 进行排序
        $query->orderBy($sortBy, $orderBy);

        // 获取分页数据
        $records = $query->paginate($itemsPerPage, ['*'], 'page', $page);

        // 返回响应数据
        return response()->json([
            'records' => $records->items(),
            'total' => $records->total(),
            'current_page' => $records->currentPage(),
            'per_page' => $records->perPage(),
        ]);
    }
}