<?php

namespace App\Http\Controllers\Order;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\Order;
use App\Models\Company;
use App\Models\Product;
use App\Models\InventoryProduct;
use App\Models\User;
use Illuminate\Http\Request;
use Carbon\Carbon;
use DB;

class OrdersController extends Controller
{
    /**
     * Display a listing of the orders.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        // 获取查询参数
        $searchQuery = $request->input('q', '');
        $searchColumn = $request->input('column', 'all');
        $page = $request->input('page', 1);
        $itemsPerPage = $request->input('itemsPerPage', 10);
        $sortBy = $request->input('sortBy', 'created_at');
        $orderBy = $request->input('orderBy', 'desc');

        $user = $request->user();

        // 查询订单数据
        $query = Order::query()->where('company_id', $user->company_id);

        // 如果有搜索查询，则按条件过滤
        if ($searchQuery) {
            $query->where(function($q) use ($searchQuery, $searchColumn) {
                if ($searchColumn === 'all') {
                    $q->where('user_email', 'like', '%' . $searchQuery . '%')
                      ->orWhere('id', $searchQuery)
                      ->orWhere('status', 'like', '%' . $searchQuery . '%')
                      ->orWhere('created_at', 'like', '%' . $searchQuery . '%')
                      ->orWhere('total', $searchQuery);
                } else {
                    if (in_array($searchColumn, ['id', 'total'])) {
                        $q->where($searchColumn, $searchQuery);
                    } else {
                        $q->where($searchColumn, 'like', '%' . $searchQuery . '%');
                    }
                }
            });
        }

        // 进行排序
        $query->orderBy($sortBy, $orderBy);

        // 获取分页数据
        $orders = $query->paginate($itemsPerPage, ['*'], 'page', $page);

        $orderCounts = Order::select('status', \DB::raw('count(*) as total'))->where('company_id', $user->company_id)->groupBy('status')->get();

        $orderStatuses = [];

        foreach ($orderCounts as $orderCount) {
            $orderStatuses[$orderCount['status']] = $orderCount['total'];
        }

        // 返回响应数据
        return response()->json([
            'orders' => $orders->items(),
            'total' => $orders->total(),
            'current_page' => $orders->currentPage(),
            'per_page' => $orders->perPage(),
            'orderStatuses' => $orderStatuses
        ]);
    }

    public function getOrderDetails($id)
    {
        try {
            $user = Auth::user();

            $order = Order::where('company_id', $user->company_id)->findOrFail($id);

            $inventoryProducts = InventoryProduct::where('order_id', $order->id)->where('inventory_type', 'in')->get();

            $totalQuantity = 0;

            foreach($inventoryProducts as $key => $inventoryProduct) {
                $totalQuantity += $inventoryProduct->quantity;
            }

            $company = company::find($user->company_id);

            $carbon = Carbon::createFromFormat('Y-m-d H:i:s', $order->created_at);
            $carbon->addMinutes(10);

            return response()->json([
                'success' => true,  
                'order' => [
                    'id' => $order->id,
                    'total' => $order->total,
                    'use_balance' => $order->use_balance,
                    'quantity' => $totalQuantity,
                    'user_email' => $order->user_email,
                    'company_name' => $company->name,
                    'payment_method' => $order->payment_method,
                    'status' => $order->status,
                    'created_at' => Carbon::parse($order->created_at)->toDateTimeString(),
                    'cancel_timestamp' => $carbon->timestamp,
                ],
                'inventoryProducts' => $inventoryProducts 
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to Retrieve Order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function cancelOrder($id)
    {
        try {
            $user = Auth::user();

            DB::beginTransaction();

            $order = Order::findOrFail($id);

            if($user->group_id == 99) {
                $user = User::findOrFail($order->user_id);
            }

            $orderProducts = InventoryProduct::where('order_id', $id)->where('inventory_type', 'in')->where('company_id', $user->company_id)->get();

            $orderInventoryArr = [];

            foreach($orderProducts as $orderProduct) {
                $orderInventoryKey = $orderProduct->product_id . '_' . $orderProduct->width . '_' . $orderProduct->height;
                if(!isset($orderInventoryArr[$orderInventoryKey])) {
                    $orderInventoryArr[$orderInventoryKey]['product_id'] = $orderProduct->product_id;
                    $orderInventoryArr[$orderInventoryKey]['width'] = $orderProduct->width;
                    $orderInventoryArr[$orderInventoryKey]['height'] = $orderProduct->height;
                    $orderInventoryArr[$orderInventoryKey]['quantity'] = 0;
                }

                $orderInventoryArr[$orderInventoryKey]['quantity'] += $orderProduct->quantity;
            }

            foreach($orderInventoryArr as $key => $orderInventory) {
                $inventoryProductTotal = InventoryProduct::findInventoryProductsTotal($orderInventory, $user->company_id);
                if($inventoryProductTotal < $orderInventory['quantity']) {
                    throw new \Exception('You have already used the inventory purchased with this order, so the order cannot be canceled.');
                }
            }

            foreach($orderInventoryArr as $orderInventory) {
                Product::where('id', $orderInventory['product_id'])->update(['quantity' => DB::raw('quantity + ' . $orderInventory['quantity'])]);
            }

            InventoryProduct::where('order_id', $id)->where('inventory_type', 'in')->where('quantity', '>', 0)->update([
                'status' => 'Canceled',
                'quantity' => 0
            ]);

            if($user->company_id == $order->company_id) {
                $companyAdmin = User::where('company_id', $user->company_id)->where('group_id', 10)->first();
                $companyAdmin->balance += $order->use_balance;
                $companyAdmin->save();
            }

            $order->status = 'Canceled';
            $order->save();

            DB::commit();

            return response()->json([
                'success' => true,  
                'status' => $order->status,
            ]);
               

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage() ?? 'Failed to cancel order',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
