import { defineStore } from 'pinia'
import { ref } from 'vue'
import { request } from '@/utils/axios'

export const useUserStore = defineStore('user', () => {
  // State
  const user = ref(null) // Changed from [] to null as initial state
  const isAuthenticated = ref(false)

  // Actions
  const setUser = (userData) => {
    user.value = userData
    isAuthenticated.value = true
  }

  const clearUser = () => {
    user.value = null
    isAuthenticated.value = false
  }

  // Fetch user information
  const fetchUser = async () => {
    try {
      const response = await request.get('/user') // Default user info endpoint for Laravel Sanctum
      
      if (response) {
        setUser(response)
        return response
      }
      return null
    } catch (error) {
      console.error('Failed to fetch user info:', error)
      clearUser()
      return null
    }
  }

  // Login
  const login = async (credentials) => {
    try {
      // 先获取CSRF令牌
      await request.get('/sanctum/csrf-cookie')
      const response = await request.post('/login', credentials)
      
      if (response.user) {
        setUser(response.user)
        return { success: true }
      }
      
      // 修改这里的错误返回格式
      return { 
        success: false, 
        message: response.message,
        errors: response.errors // 添加这行，传递验证错误信息
      }
    } catch (error) {
      console.error('Login failed:', error)
      // 如果是验证错误（422），返回后端的错误信息
      if (error.response?.status === 422) {
        return {
          success: false,
          message: error.response.data.message,
          errors: error.response.data.errors
        }
      }
      
      return { 
        success: false, 
        message: 'Login request failed',
        errors: {}
      }
    }
  }

  // Add loading state
  const loading = ref(false)

  // Logout function
  const logout = async () => {
    try {
      await request.post('/logout')
      clearUser()
      return { success: true }
    } catch (error) {
      console.error('Logout failed:', error)
      return { 
        success: false, 
        message: 'Logout failed'
      }
    }
  }

  return {
    // State
    user,
    isAuthenticated,
    loading,
    // Actions
    setUser,
    clearUser,
    fetchUser,
    login, 
    logout
  }
})